<?php

namespace app\models;

use Yii;
use yii\db\ActiveRecord;
use yii\behaviors\TimestampBehavior;

/**
 * Class Application
 * @package app\models
 * @property int $id
 * @property int $user_id
 * @property int $tour_id
 * @property string $tour_date
 * @property int $participants_count
 * @property string $contact_phone
 * @property string $comment
 * @property string $payment_method
 * @property string $wishes
 * @property string $status
 * @property string $admin_comment
 * @property int $created_at
 * @property int $updated_at
 * 
 * @property User $user
 * @property Tour $tour
 */
class Application extends ActiveRecord
{
    const STATUS_NEW = 'new';
    const STATUS_PROCESSING = 'processing';
    const STATUS_ACCEPTED = 'accepted';
    const STATUS_REJECTED = 'rejected';
    const STATUS_CANCELLED = 'cancelled';

    const PAYMENT_CASH = 'cash';
    const PAYMENT_CARD = 'card';
    const PAYMENT_TRANSFER = 'transfer';

    /**
     * {@inheritdoc}
     */
    public static function tableName()
    {
        return 'application'; // Название таблицы из миграции
    }

    /**
     * {@inheritdoc}
     */
    public function rules()
    {
        return [
            [['user_id', 'tour_id', 'tour_date', 'participants_count'], 'required'],
            [['user_id', 'tour_id', 'participants_count'], 'integer'],
            [['tour_date'], 'date', 'format' => 'php:Y-m-d'],
            [['participants_count'], 'integer', 'min' => 1, 'max' => 20],
            [['comment', 'wishes', 'admin_comment'], 'string'],
            [['contact_phone', 'payment_method', 'status'], 'string', 'max' => 50],
            [['status'], 'default', 'value' => self::STATUS_NEW],
            [['payment_method'], 'in', 'range' => array_keys(self::getPaymentMethods())],
            [['status'], 'in', 'range' => array_keys(self::getStatuses())],
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function attributeLabels()
    {
        return [
            'id' => 'ID',
            'user_id' => 'Пользователь',
            'tour_id' => 'Тур',
            'tour_date' => 'Дата тура',
            'participants_count' => 'Количество участников',
            'contact_phone' => 'Контактный телефон',
            'comment' => 'Комментарий',
            'payment_method' => 'Способ оплаты',
            'wishes' => 'Пожелания',
            'status' => 'Статус',
            'admin_comment' => 'Комментарий администратора',
            'created_at' => 'Дата создания',
            'updated_at' => 'Дата обновления',
        ];
    }

    /**
     * Связь с пользователем
     */
    public function getUser()
    {
        return $this->hasOne(User::class, ['id' => 'user_id']);
    }

    /**
     * Связь с туром
     */
    public function getTour()
    {
        return $this->hasOne(Tour::class, ['id' => 'tour_id']);
    }

    /**
     * Получение всех статусов
     */
    public static function getStatuses()
    {
        return [
            self::STATUS_NEW => 'Новая',
            self::STATUS_PROCESSING => 'На рассмотрении',
            self::STATUS_ACCEPTED => 'Принята',
            self::STATUS_REJECTED => 'Отклонена',
            self::STATUS_CANCELLED => 'Отменена',
        ];
    }

    /**
     * Получение label статуса
     */
    public function getStatusLabel()
    {
        $statuses = self::getStatuses();
        return $statuses[$this->status] ?? $this->status;
    }

    /**
     * Получение способов оплаты
     */
    public static function getPaymentMethods()
    {
        return [
            self::PAYMENT_CASH => 'Наличные',
            self::PAYMENT_CARD => 'Банковская карта',
            self::PAYMENT_TRANSFER => 'Банковский перевод',
        ];
    }

    /**
     * Получение label способа оплаты
     */
    public function getPaymentMethodLabel()
    {
        $methods = self::getPaymentMethods();
        return $methods[$this->payment_method] ?? $this->payment_method;
    }

    /**
     * Форматированная дата тура
     */
    public function getFormattedTourDate()
    {
        return date('d.m.Y', strtotime($this->tour_date));
    }

    /**
     * Форматированная дата создания
     */
    public function getFormattedCreatedAt()
    {
        return date('d.m.Y H:i', $this->created_at);
    }

    /**
     * Общая стоимость заявки
     */
    public function getTotalPrice()
    {
        if ($this->tour) {
            return $this->tour->price * $this->participants_count;
        }
        return 0;
    }

    /**
     * Форматированная общая стоимость
     */
    public function getFormattedTotalPrice()
    {
        return number_format($this->getTotalPrice(), 0, ',', ' ') . ' ₽';
    }

    /**
     * Проверка, может ли пользователь отменить заявку
     */
    public function canBeCancelled()
    {
        return $this->status === self::STATUS_NEW || 
               $this->status === self::STATUS_PROCESSING;
    }

    /**
     * Проверка, может ли пользователь оставить отзыв
     */
    public function canAddReview()
    {
        return $this->status === self::STATUS_ACCEPTED && 
               strtotime($this->tour_date) < time();
    }
}