<?php

use yii\helpers\Html;
use yii\helpers\Url;

$this->title = 'Регистрация';
?>
<div class="auth-container">
    <div class="card">
        <div class="card-header">
            <h2><i class="fas fa-user-plus"></i> <?= Html::encode($this->title) ?></h2>
        </div>
        <div class="card-body">
            <?php if (Yii::$app->session->hasFlash('success')): ?>
                <div class="alert alert-success">
                    <i class="fas fa-check-circle"></i> <?= Yii::$app->session->getFlash('success') ?>
                </div>
            <?php endif; ?>

            <?php if (Yii::$app->session->hasFlash('error')): ?>
                <div class="alert alert-danger">
                    <i class="fas fa-exclamation-circle"></i> <?= Yii::$app->session->getFlash('error') ?>
                </div>
            <?php endif; ?>

            <form id="register-form" action="<?= Url::to(['auth/register']) ?>" method="post" autocomplete="off">
                <input type="hidden" name="<?= Yii::$app->request->csrfParam ?>" value="<?= Yii::$app->request->csrfToken ?>">
                
                <div class="form-group">
                    <label for="email" class="form-label">Email *</label>
                    <input type="email" 
                           id="email" 
                           name="User[email]" 
                           class="form-control" 
                           required
                           placeholder="example@mail.ru"
                           value="<?= isset($model->email) ? Html::encode($model->email) : '' ?>">
                    <div class="invalid-feedback">Введите корректный email адрес</div>
                </div>

                <div class="row">
                    <div class="col-md-6">
                        <div class="form-group">
                            <label for="password" class="form-label">Пароль *</label>
                            <input type="password" 
                                   id="password" 
                                   name="User[password]" 
                                   class="form-control" 
                                   required
                                   minlength="7"
                                   placeholder="Минимум 7 символов">
                            <div class="invalid-feedback">Пароль должен содержать минимум 7 символов</div>
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="form-group">
                            <label for="password_repeat" class="form-label">Повтор пароля *</label>
                            <input type="password" 
                                   id="password_repeat" 
                                   name="User[password_repeat]" 
                                   class="form-control" 
                                   required
                                   placeholder="Повторите пароль">
                            <div class="invalid-feedback">Пароли не совпадают</div>
                        </div>
                    </div>
                </div>

                <div class="form-group">
                    <label for="full_name" class="form-label">ФИО *</label>
                    <input type="text" 
                           id="full_name" 
                           name="User[full_name]" 
                           class="form-control" 
                           required
                           pattern="[А-Яа-яЁё\s]+"
                           placeholder="Иванов Иван Иванович"
                           value="<?= isset($model->full_name) ? Html::encode($model->full_name) : '' ?>">
                    <div class="invalid-feedback">ФИО должно содержать только русские буквы и пробелы</div>
                </div>

                <div class="row">
                    <div class="col-md-6">
                        <div class="form-group">
                            <label for="phone" class="form-label">Телефон *</label>
                            <input type="tel" 
                                   id="phone" 
                                   name="User[phone]" 
                                   class="form-control phone-input" 
                                   required
                                   pattern="\+7\(\d{3}\)\d{3}-\d{2}-\d{2}"
                                   placeholder="+7(999)123-45-67"
                                   value="<?= isset($model->phone) ? Html::encode($model->phone) : '' ?>">
                            <div class="invalid-feedback">Телефон должен быть в формате +7(XXX)XXX-XX-XX</div>
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="form-group">
                            <label for="passport" class="form-label">Паспорт *</label>
                            <input type="text" 
                                   id="passport" 
                                   name="User[passport]" 
                                   class="form-control" 
                                   required
                                   placeholder="1234 567890"
                                   value="<?= isset($model->passport) ? Html::encode($model->passport) : '' ?>">
                            <div class="invalid-feedback">Введите серию и номер паспорта</div>
                        </div>
                    </div>
                </div>

                <div class="form-group">
                    <label for="address" class="form-label">Адрес проживания *</label>
                    <textarea id="address" 
                              name="User[address]" 
                              class="form-control" 
                              required
                              rows="3"
                              placeholder="г. Екатеринбург, ул. Ленина, д. 1, кв. 1"><?= isset($model->address) ? Html::encode($model->address) : '' ?></textarea>
                    <div class="invalid-feedback">Введите адрес проживания</div>
                </div>

                <div class="form-group">
                    <div class="checkbox">
                        <input type="checkbox" 
                               id="agree_terms" 
                               name="User[agree_terms]" 
                               value="1" 
                               required>
                        <label for="agree_terms">
                            Я согласен с <a href="#" onclick="showTerms()">правилами регистрации</a>
                        </label>
                        <div class="invalid-feedback">Вы должны согласиться с правилами регистрации</div>
                    </div>
                </div>

                <div class="form-group">
                    <button type="submit" class="btn btn-primary btn-block btn-lg">
                        <i class="fas fa-user-plus"></i> Зарегистрироваться
                    </button>
                </div>

                <div class="text-center">
                    <p>Уже зарегистрированы? <a href="<?= Url::to(['auth/login']) ?>">Войти в систему</a></p>
                    <p><a href="<?= Url::to(['site/index']) ?>">Вернуться на главную</a></p>
                </div>
            </form>
        </div>
    </div>
</div>

<div id="termsModal" class="modal">
    <div class="modal-content">
        <div class="modal-header">
            <h3>Правила регистрации</h3>
            <span class="modal-close" onclick="closeModal()">&times;</span>
        </div>
        <div class="modal-body">
            <p>Регистрируясь на нашем портале, вы соглашаетесь со следующими правилами:</p>
            <ol>
                <li>Вы предоставляете достоверную информацию о себе</li>
                <li>Нельзя регистрировать несколько аккаунтов на одного человека</li>
                <li>Вы несете ответственность за сохранность своих учетных данных</li>
                <li>Запрещается использовать портал в противоправных целях</li>
                <li>Мы обязуемся хранить ваши данные в соответствии с законом о персональных данных</li>
            </ol>
            <p class="text-muted">Нажимая кнопку "Зарегистрироваться", вы подтверждаете свое согласие с этими правилами.</p>
        </div>
        <div class="modal-footer">
            <button type="button" class="btn btn-secondary" onclick="closeModal()">Закрыть</button>
        </div>
    </div>
</div>

<style>
.auth-container {
    max-width: 800px;
    margin: 40px auto;
    padding: 20px;
}

.card {
    background: white;
    border-radius: 15px;
    box-shadow: 0 8px 30px rgba(0, 0, 0, 0.1);
    overflow: hidden;
    transition: transform 0.3s ease;
}

.card:hover {
    transform: translateY(-5px);
}

.card-header {
    background: linear-gradient(45deg, #2c3e50, #34495e);
    color: white;
    padding: 20px;
}

.card-body {
    padding: 30px;
}

.form-group {
    margin-bottom: 25px;
}

.form-label {
    display: block;
    margin-bottom: 8px;
    font-weight: 600;
    color: #2c3e50;
    font-size: 14px;
}

.form-control {
    width: 100%;
    padding: 12px 15px;
    border: 2px solid #e0e0e0;
    border-radius: 8px;
    font-size: 16px;
    transition: all 0.3s ease;
    background: #f8f9fa;
}

.form-control:focus {
    border-color: #3498db;
    background: white;
    box-shadow: 0 0 0 3px rgba(52, 152, 219, 0.1);
    outline: none;
}

.form-control.is-invalid {
    border-color: #e74c3c;
}

.invalid-feedback {
    display: none;
    color: #e74c3c;
    font-size: 14px;
    margin-top: 5px;
}

.form-control.is-invalid + .invalid-feedback {
    display: block;
}

.checkbox {
    display: flex;
    align-items: center;
    gap: 10px;
}

.checkbox input[type="checkbox"] {
    width: 18px;
    height: 18px;
    cursor: pointer;
}

.checkbox label {
    cursor: pointer;
    margin-bottom: 0;
}

.btn-block {
    width: 100%;
}

.text-center {
    text-align: center;
}

.text-center a {
    color: #3498db;
    text-decoration: none;
}

.text-center a:hover {
    text-decoration: underline;
}

/* Модальное окно */
.modal {
    display: none;
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: rgba(0, 0, 0, 0.5);
    z-index: 1000;
    opacity: 0;
    transition: opacity 0.3s ease;
}

.modal.show {
    display: flex;
    align-items: center;
    justify-content: center;
    opacity: 1;
}

.modal-content {
    background: white;
    border-radius: 10px;
    max-width: 600px;
    width: 90%;
    max-height: 80vh;
    overflow-y: auto;
    animation: modalSlideIn 0.3s ease;
}

@keyframes modalSlideIn {
    from {
        transform: translateY(-50px);
        opacity: 0;
    }
    to {
        transform: translateY(0);
        opacity: 1;
    }
}

.modal-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 20px;
    border-bottom: 1px solid #eee;
}

.modal-close {
    font-size: 24px;
    cursor: pointer;
    color: #95a5a6;
    transition: color 0.3s ease;
}

.modal-close:hover {
    color: #e74c3c;
}

.modal-body {
    padding: 20px;
}

.modal-footer {
    padding: 20px;
    border-top: 1px solid #eee;
    text-align: right;
}

/* Сетка для строк */
.row {
    display: flex;
    flex-wrap: wrap;
    margin: 0 -10px;
}

.col-md-6 {
    flex: 0 0 50%;
    max-width: 50%;
    padding: 0 10px;
}

@media (max-width: 768px) {
    .col-md-6 {
        flex: 0 0 100%;
        max-width: 100%;
        margin-bottom: 15px;
    }
    
    .auth-container {
        margin: 20px auto;
        padding: 10px;
    }
    
    .card-body {
        padding: 20px;
    }
}
</style>

<script>
// Маска для телефона
function phoneMask(input) {
    let value = input.value.replace(/\D/g, '');
    let formatted = '';
    
    if (value.length > 0) {
        formatted = '+7(';
    }
    if (value.length > 1) {
        formatted += value.substring(1, 4);
    }
    if (value.length >= 4) {
        formatted += ')';
    }
    if (value.length >= 4) {
        formatted += value.substring(4, 7);
    }
    if (value.length >= 7) {
        formatted += '-';
    }
    if (value.length >= 7) {
        formatted += value.substring(7, 9);
    }
    if (value.length >= 9) {
        formatted += '-';
    }
    if (value.length >= 9) {
        formatted += value.substring(9, 11);
    }
    
    input.value = formatted;
}

// Валидация формы
function validateForm() {
    let isValid = true;
    const form = document.getElementById('register-form');
    const inputs = form.querySelectorAll('input[required], textarea[required]');
    
    // Сброс предыдущих ошибок
    inputs.forEach(input => {
        input.classList.remove('is-invalid');
    });
    
    // Проверка email
    const email = document.getElementById('email');
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    if (!emailRegex.test(email.value)) {
        email.classList.add('is-invalid');
        isValid = false;
    }
    
    // Проверка пароля
    const password = document.getElementById('password');
    if (password.value.length < 7) {
        password.classList.add('is-invalid');
        isValid = false;
    }
    
    // Проверка подтверждения пароля
    const passwordRepeat = document.getElementById('password_repeat');
    if (password.value !== passwordRepeat.value) {
        passwordRepeat.classList.add('is-invalid');
        isValid = false;
    }
    
    // Проверка ФИО (только кириллица)
    const fullName = document.getElementById('full_name');
    const cyrillicRegex = /^[А-Яа-яЁё\s]+$/;
    if (!cyrillicRegex.test(fullName.value)) {
        fullName.classList.add('is-invalid');
        isValid = false;
    }
    
    // Проверка телефона
    const phone = document.getElementById('phone');
    const phoneRegex = /^\+7\(\d{3}\)\d{3}-\d{2}-\d{2}$/;
    if (!phoneRegex.test(phone.value)) {
        phone.classList.add('is-invalid');
        isValid = false;
    }
    
    // Проверка согласия с правилами
    const agreeTerms = document.getElementById('agree_terms');
    if (!agreeTerms.checked) {
        agreeTerms.classList.add('is-invalid');
        isValid = false;
    }
    
    // Проверка остальных обязательных полей
    inputs.forEach(input => {
        if (!input.value.trim()) {
            input.classList.add('is-invalid');
            isValid = false;
        }
    });
    
    if (!isValid) {
        // Прокрутка к первой ошибке
        const firstError = form.querySelector('.is-invalid');
        if (firstError) {
            firstError.scrollIntoView({ behavior: 'smooth', block: 'center' });
            firstError.focus();
        }
        
        // Анимация ошибки
        form.style.animation = 'none';
        setTimeout(() => {
            form.style.animation = 'shake 0.5s ease';
        }, 10);
    }
    
    return isValid;
}

// Анимация shake
const style = document.createElement('style');
style.textContent = `
    @keyframes shake {
        0%, 100% { transform: translateX(0); }
        10%, 30%, 50%, 70%, 90% { transform: translateX(-5px); }
        20%, 40%, 60%, 80% { transform: translateX(5px); }
    }
`;
document.head.appendChild(style);

// Модальное окно
function showTerms() {
    const modal = document.getElementById('termsModal');
    modal.style.display = 'flex';
    setTimeout(() => {
        modal.classList.add('show');
    }, 10);
    return false;
}

function closeModal() {
    const modal = document.getElementById('termsModal');
    modal.classList.remove('show');
    setTimeout(() => {
        modal.style.display = 'none';
    }, 300);
}

// Инициализация при загрузке
document.addEventListener('DOMContentLoaded', function() {
    // Маска для телефона
    const phoneInput = document.getElementById('phone');
    if (phoneInput) {
        phoneInput.addEventListener('input', function() {
            phoneMask(this);
        });
        
        phoneInput.addEventListener('blur', function() {
            const phoneRegex = /^\+7\(\d{3}\)\d{3}-\d{2}-\d{2}$/;
            if (this.value && !phoneRegex.test(this.value)) {
                this.classList.add('is-invalid');
            }
        });
    }
    
    // Валидация при вводе
    const form = document.getElementById('register-form');
    if (form) {
        const inputs = form.querySelectorAll('input, textarea');
        inputs.forEach(input => {
            input.addEventListener('blur', function() {
                if (this.hasAttribute('required') && !this.value.trim()) {
                    this.classList.add('is-invalid');
                } else {
                    this.classList.remove('is-invalid');
                }
            });
            
            input.addEventListener('input', function() {
                this.classList.remove('is-invalid');
            });
        });
        
        // Обработка отправки формы
        form.addEventListener('submit', function(e) {
            if (!validateForm()) {
                e.preventDefault();
            } else {
                // Показать загрузку
                const submitBtn = this.querySelector('button[type="submit"]');
                const originalText = submitBtn.innerHTML;
                submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Регистрация...';
                submitBtn.disabled = true;
                
                // Восстановить кнопку если форма не отправится
                setTimeout(() => {
                    submitBtn.innerHTML = originalText;
                    submitBtn.disabled = false;
                }, 3000);
            }
        });
    }
    
    // Закрытие модалки по клику вне
    const modal = document.getElementById('termsModal');
    if (modal) {
        modal.addEventListener('click', function(e) {
            if (e.target === this) {
                closeModal();
            }
        });
    }
});
</script>